<?php

/**
 * @file media_youtube/themes/media_youtube.theme.inc
 *
 * Theme and preprocess functions for Media: YouTube.
 */

/**
 * Preprocess function for theme('media_youtube_video').
 */
function media_youtube_preprocess_media_youtube_video(&$variables) {
  // Build the URI.
  $wrapper = file_stream_wrapper_get_instance_by_uri($variables['uri']);
  if ($wrapper instanceof MediaReadOnlyStreamWrapper) {
    $parts = $wrapper->get_parameters();
    if (isset($parts['v'])) {
      $variables['embed_type'] = 'video';
      $variables['video_id'] = check_plain($parts['v']);
      $embed_path = '/embed/' . $variables['video_id'];
    }
    elseif (isset($parts['l'])) {
      $variables['embed_type'] = 'playlist';
      $variables['video_id'] = check_plain($parts['l']);
      $embed_path = '/embed/videoseries';
    }
  }
  else {
    // This happens when stream wrappers are not yet initialized. This is
    // normally only encountered when creating content during profile install
    // using drush make. At that point, video_id is irrelevant anyway.
    $variables['video_id'] = '';
  }

  // Checked existing function.
  if(function_exists('file_uri_to_object')) {
    // Make the file object available.
    $file_object = file_uri_to_object($variables['uri'], TRUE);
  }
  else {
    $file_object = media_youtube_file_uri_to_object($variables['uri']);
  }

  // Parse options and build the query string. Only add the option to the query
  // array if the option value is not default. Be careful, depending on the
  // wording in media_youtube.formatters.inc, TRUE may be query=0.
  // @see https://developers.google.com/youtube/player_parameters#Parameters
  $query = array();

  // Make css z-index work with flash object. Must be the first parameter.
  $query['wmode'] = 'opaque';

  //YouTube video controls, on or off.
  if (isset($variables['options']['controls'])) {
    //on or off (TRUE/FALSE) depending on what is stored in $variables['options']['controls'].
    $query['controls'] = $variables['options']['controls'];
  } else {
    //on
    $query['controls'] = TRUE;
  }
  // These queries default to 0. If the option is true, set value to 1.
  foreach (array('autoplay', 'enablejsapi', 'loop', 'modestbranding') as $option) {
    if (isset($variables['options'][$option]) && $variables['options'][$option]) {
      $query[$option] = 1;
    }
  }
  if ($variables['options']['enablejsapi']) {
    // Add a query ID and identical html ID if js API is set.
    $query['playerapiid'] = drupal_html_id('media-youtube-' . $variables['video_id']);
    $variables['api_id_attribute'] = 'id="' . $query['playerapiid'] . '" ';

    // Add the origin for improved security
    $variables['options']['origin'] ? $query['origin'] = $variables['options']['origin'] : '';
  }
  else {
    $variables['api_id_attribute'] = NULL;
  }

  // Currently, loop only works with a playlist. Make fake playlist out of a
  // single video.
  // @see https://developers.google.com/youtube/player_parameters#loop
  if ($variables['options']['loop']) {
    $query['playlist'] = $variables['video_id'];
  }

  // These queries default to 1. If the option is false, set value to 0.
  foreach (array('rel', 'showinfo') as $option) {
    if (!$variables['options'][$option]) {
      $query[$option] = 0;
    }
  }

  // These queries default to 1. Option wording is reversed, so if the option
  // is true, set value to 0.
  // (None right now.)

  // Strings.
  if ($variables['options']['theme'] != 'dark') {
    $query['theme'] = $variables['options']['theme'];
  }
  if ($variables['options']['color'] != 'red') {
    $query['color'] = $variables['options']['color'];
  }
  if ($variables['options']['autohide'] != '2') {
    $query['autohide'] = $variables['options']['autohide'];
  }
  if ($variables['options']['captions'] > '0') {
    // Add captions parameters to the query.
    $query['cc_load_policy'] = '1';
    if ($variables['options']['captions'] > '1') {
      global $language;
      // We can specify a default language for captions.
      if ($language->language != LANGUAGE_NONE) {
        $query['cc_lang_pref'] = $language->language;
      }
    }
  }

  if ($variables['options']['protocol_specify']) {
    $protocol = $variables['options']['protocol'];
  }
  else {
    $protocol = 'https:';
  }

  // Non-query options.
  if ($variables['options']['nocookie']) {
    $url_base = 'youtube-nocookie.com';
  }
  else {
    $url_base = 'youtube.com';
  }

  // Add some options as their own template variables.
  foreach (array('width', 'height') as $theme_var) {
    $variables[$theme_var] = $variables['options'][$theme_var];
  }

  // Do something useful with the overridden attributes from the file
  // object. We ignore alt and style for now.
  if (isset($variables['options']['attributes']['class'])) {
    if (is_array($variables['options']['attributes']['class'])) {
      $variables['classes_array'] = array_merge($variables['classes_array'], $variables['options']['attributes']['class']);
    }
    else {
      // Provide nominal support for Media 1.x.
      $variables['classes_array'][] = $variables['options']['attributes']['class'];
    }
  }

  // Add template variables for accessibility.
  $variables['title'] = check_plain($file_object->filename);
  // @TODO: Find an easy/ not too expensive way to get the YouTube description
  // to use for the alternative content.
  $variables['alternative_content'] = t('Video of @title', array('@title' => $variables['title']));

  if (isset($parts['l'])) {
    $query['list'] = $parts['l'];
  }
  // Build the iframe URL with options query string.
  $variables['url'] = url($protocol . '//www.' . $url_base . $embed_path, array('query' => $query, 'external' => TRUE));
}

/**
 * Helping function.
 */
function media_youtube_file_uri_to_object($uri) {
  $uri = file_stream_wrapper_uri_normalize($uri);
  $files = entity_load('file', FALSE, array('uri' => $uri));
  $file = !empty($files) ? reset($files) : FALSE;
  if (!$file) {
    global $user;
    $file = new stdClass();
    $file->uid = $user->uid;
    $file->filename = basename($uri);
    $file->uri = $uri;
    $file->filemime = file_get_mimetype($uri);
    // This is gagged because some uris will not support it.
    $file->filesize = @filesize($uri);
    $file->timestamp = REQUEST_TIME;
    $file->status = FILE_STATUS_PERMANENT;
  }
  return $file;
}
